#ifndef smplRN_CK
#define smplRN_CK

namespace smplRN {

#include "math.h"


/********************************************

	B.D.Ripley, "Stochastic Simulation", 1987.
		See Table 2.4
            a = 69069;  c = 1;

	The period is of order 10 to the 9th power
                  i.e. 4giga.
		  See, Theorem 2.1 in Ripley (1987)

*********************************************/

class smplrn {
	public:
		const unsigned __int32 a, c;
		unsigned __int32 seed;

		smplrn();
		smplrn( unsigned __int32 s );

		virtual ~smplrn(){ }

		unsigned __int32 getSeed(){
			return seed;
		}

		double uni();     //    0 < uni() < 1
		double normal();  //    Standard normal distribution
		void normalp( double & x1, double & x2 );  //  x1 and x2 are independently normally distributed
};

	smplrn::smplrn(): a(69069), c(1), seed(1){ }

	smplrn::smplrn( unsigned __int32 s ): a(69069), c(1), seed(s){ }


	double smplrn::uni(){
		seed = a * seed + c;
		double v = (double(seed) + 1.0) / (double(0xFFFFFFFF) + 2.0);

		return v;
	}

	/*********************************************************************

		A Rejection Polar Method for Normal Variates.

				cf J. E. Gentle (2003). 
	"Random Number Generation and Monte Carlo Methods, 2nd ed.", p. 173.

	**********************************************************************/
	double smplrn::normal(){
		double v1, v2, r2;
		do {
			v1 = 2.0 * uni() - 1.0; v2 = 2.0 * uni() - 1.0;
			r2 = (v1 * v1) + (v2 * v2);
		} while (r2 >= 1.0) ;
		return v1 * sqrt( -2.0 * log(r2) / r2 );
	}

	void smplrn::normalp( double & x1, double & x2 ){
		double v1, v2, r2;
		do {
			v1 = 2.0 * uni() - 1.0; v2 = 2.0 * uni() - 1.0;
			r2 = (v1 * v1) + (v2 * v2);
		} while (r2 >= 1.0);
		x1 = v1 * sqrt( -2.0 * log(r2) / r2 );
		x2 = v2 * sqrt( -2.0 * log(r2) / r2 );
		return;
	}

}

#endif
