//   Version 2008.08.25
//   Version 2008.10.10,  modified for multi-thread programming


#ifndef MyRNKISS_CK
#define MyRNKISS_CK

#include <math.h>

#include "mylib.h"

using namespace System;

namespace MyRNKISS {

/****************************************************************

                    The Kiss Algorithm

              See   C.P.Robert and G.Casella
             "Monte Carlo Statistical Methods"
                 1999, Springer-Verlag
                         Pp. 39-43.

******************************************************************/

	class myrn {
    	private:
			unsigned __int32 x, y, z;
			int icycle, ibias;

	    public:
			myrn();
			myrn( int ic, int ib );
			myrn( unsigned __int32 xv, unsigned __int32 yv, unsigned __int32 zv );
		    virtual ~myrn() { }; 

		    unsigned __int32 seek_x(){ return x;};
			unsigned __int32 seek_y(){ return y;};
			unsigned __int32 seek_z(){ return z;};

			void init();
			void init( unsigned __int32 xv, unsigned __int32 yv, unsigned __int32 zv );
			void Init( int ic, int ib );

			double uni0();                           
			double uni();                             //   0.0 < uni()  <= 1.0
			double normal();                          //   Standard Normal Distribution
													  //   Rejection polar method for normal variate
			double normalMS(double m, double s);      //    m : mean,   s : standard deviation  
			void   normalPair( double & n1, double & n2 );   //    pair of independent random numbers of  
			                                                 //             the standard normal distribution 
    };



	myrn::myrn() {
		x = 0x95555555;
		y = 0x56666666;
		z = 0x39999999;
		icycle = 0;
		ibias = 0;
	}

	myrn::myrn( int ic, int ib ){
		x = 0x95555555;
		y = 0x56666666;
		z = 0x39999999;
		icycle = ic;
		ibias = ib;
		if (ib > 0)
			for (int i = 0; i < ib; i++) uni0();
	}
	

	myrn::myrn( unsigned __int32 xv, unsigned __int32 yv, unsigned __int32 zv ){
		x = xv;
		y = yv;
		z = zv;
		icycle = 0;
		ibias = 0;
	}


	void myrn::init(){
		x = 0x95555555;
		y = 0x56666666;
		z = 0x39999999;
		icycle = 0;
		ibias = 0;
	}


	void myrn::init( unsigned __int32 xv, unsigned __int32 yv, unsigned __int32 zv ){
		x = xv;
		y = yv;
		z = zv;
		icycle =0;
		ibias = 0;
	}

	void myrn::Init( int ic, int ib ){
		icycle = ic;
		ibias = ib;
		if (ib > 0)
			for (int i = 0; i < ib; i++) uni0();

	}


	//
	//     cf. Robert and Casella (1999) "Monte Carlo Statistical Methods", 
	//                                    Algorithm A.3
	//
	//                     0.0 <  uni()  <  1.0      
	//
	double myrn::uni0() {
       const double den = ((double)0xFFFFFFFF) + 2.0;

		x = 69069 * x + 23606797;
		y ^= y << 17;
		y ^= y >> 15;
		z = (z ^ (z << 18)) & 0x7FFFFFFF;
		z ^= z >> 13;

		unsigned __int32 v = x + y + z;

		return (((double)v) + 1.0) / den;
	}


	double myrn::uni() {
		for (int i = 0; i < icycle-1; i++) uni0();

		return uni0();
	}



    //        Standard Normal Distribution
	//        Rejection polar method for normal variate
	double myrn::normal(){
		double v1, v2, w;
		do {
    		v1 = 2.0 * uni() - 1.0;
			v2 = 2.0 * uni() - 1.0;
			w  = MyLib::sqr(v1) + MyLib::sqr(v2);
		} while (((w >= 1.0) || (w <= 0.0)));

		double c = Math::Sqrt(-2.0 * Math::Log(w)/w);

		return c * v1;
	}


    //    Normal Distribution of mean m  and standard deviation s
	double myrn::normalMS(double m, double s){
			double v = s * normal() + m;
			return  v;
	}


    //    A Pair of Independent Random Numbers of 
	//      the Standard Normal Distribution 
	void  myrn::normalPair( double & n1, double & n2 ){
			double v1, v2, w, c;
			do {
				  v1 = 2.0 * uni() - 1.0;
				  v2 = 2.0 * uni() - 1.0;
				  w  = MyLib::sqr(v1) + MyLib::sqr(v2);
			} while (((w >= 1.0) || (w <= 0.0)));

			c = Math::Sqrt(-2.0 * Math::Log(w) / w);

			n1 = c * v1;
			n2 = c * v2;
	}


}

#endif
